/*
 * Decompiled with CFR 0.152.
 */
package org.luwrain.linux.services;

import java.io.IOException;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.luwrain.core.HookContainer;
import org.luwrain.core.Log;
import org.luwrain.core.Luwrain;
import org.luwrain.linux.BashProcess;
import org.luwrain.script.Hooks;
import org.luwrain.script.core.MapScriptObject;

public final class UdisksCliMonitor
implements BashProcess.Listener,
AutoCloseable {
    private static final String LOG_COMPONENT = "udisks";
    private static final Pattern RE_ADDED = Pattern.compile("^\\d\\d:\\d\\d:\\d\\d\\.\\d\\d\\d:\\sAdded\\s(.*)$");
    private static final Pattern RE_REMOVED = Pattern.compile("^\\d\\d:\\d\\d:\\d\\d\\.\\d\\d\\d:\\sRemoved\\s(.*)$");
    private static final Pattern RE_PROP_CHANGED = Pattern.compile("^\\d\\d:\\d\\d:\\d\\d\\.\\d\\d\\d:\\s*([^:]+):\\s*(.*):\\s*Properties Changed\\s*$");
    private static final String OBJ_DRIVES = "/org/freedesktop/UDisks2/drives/";
    private static final String OBJ_BLOCK = "/org/freedesktop/UDisks2/block_devices/";
    private static final String OBJ_MANAGER = "/org/freedesktop/UDisks2/Manager";
    private static final String IFACE_DRIVE = "org.freedesktop.UDisks2.Drive";
    private static final String IFACE_BLOCK = "org.freedesktop.UDisks2.Block";
    private static final String IFACE_FILESYSTEM = "org.freedesktop.UDisks2.Filesystem";
    private static final String PREFIX_REMOVABLE = "Removable:";
    private static final String PREFIX_EJECTABLE = "Ejectable:";
    private static final String PREFIX_SIZE = "Size:";
    private static final String PREFIX_MODEL = "Model:";
    private static final String PREFIX_VENDOR = "Vendor:";
    private static final String PREFIX_DEVICE = "Device:";
    private static final String PREFIX_DRIVE = "Drive:";
    private static final String PREFIX_FS_TYPE = "IdType:";
    private static final String PREFIX_MOUNT_POINTS = "MountPoints:";
    private final Luwrain luwrain;
    private final BashProcess p;
    private boolean closed = true;
    private final Map<String, Disk> disks = new HashMap<String, Disk>();
    private final Map<String, BlockDev> blockDevs = new HashMap<String, BlockDev>();
    private Disk activeDisk = null;
    private BlockDev activeBlockDev = null;
    private String activeIface = null;

    public UdisksCliMonitor(Luwrain luwrain) throws IOException {
        this.luwrain = luwrain;
        this.init();
        this.p = new BashProcess("udisksctl monitor", null, EnumSet.noneOf(BashProcess.Flags.class), this);
        this.p.run();
    }

    @Override
    public synchronized void close() {
        if (this.closed) {
            return;
        }
        Log.debug((String)LOG_COMPONENT, (String)"stopping udisksctl monitor");
        this.closed = true;
        this.p.stop();
    }

    public synchronized void enumRemovableBlockDevices(Consumer<Map<String, Object>> consumer) {
        for (Map.Entry<String, BlockDev> e : this.blockDevs.entrySet()) {
            Disk disk;
            if (!e.getValue().isReady() || (disk = this.disks.get(e.getValue().drive.replaceAll("'", ""))) == null) continue;
            HashMap<String, Object> m = new HashMap<String, Object>(e.getValue().createAttrMap());
            boolean removable = disk.removable != null && disk.removable.toLowerCase().equals("true");
            boolean ejectable = disk.ejectable != null && disk.ejectable.toLowerCase().equals("true");
            m.put("removable", removable);
            m.put("ejectable", ejectable);
            consumer.accept(m);
        }
    }

    @Override
    public synchronized void onOutputLine(String line) {
        try {
            Matcher m = RE_ADDED.matcher(line);
            if (m.find()) {
                String obj = m.group(1).trim();
                this.activeDisk = null;
                this.activeBlockDev = null;
                this.activeIface = null;
                if (obj.startsWith(OBJ_DRIVES)) {
                    this.activeDisk = new Disk(obj);
                    this.disks.put(obj, this.activeDisk);
                    Log.debug((String)LOG_COMPONENT, (String)("added new disk: " + obj));
                    return;
                }
                if (obj.startsWith(OBJ_BLOCK)) {
                    this.activeBlockDev = new BlockDev(obj);
                    this.blockDevs.put(obj, this.activeBlockDev);
                    Log.debug((String)LOG_COMPONENT, (String)("added new block device: " + obj));
                    return;
                }
                return;
            }
            m = RE_REMOVED.matcher(line);
            if (m.find()) {
                String obj = m.group(1).trim();
                this.activeDisk = null;
                this.activeBlockDev = null;
                this.activeIface = null;
                if (this.disks.containsKey(obj)) {
                    Disk disk = this.disks.get(obj);
                    this.disks.remove(obj);
                    Log.debug((String)LOG_COMPONENT, (String)("removed disk " + disk.obj));
                    Hooks.chainOfResponsibility((HookContainer)this.luwrain, (String)"luwrain.linux.disk.removed", (Object[])new Object[]{new MapScriptObject(disk.createAttrMap())});
                    return;
                }
                if (this.blockDevs.containsKey(obj)) {
                    BlockDev blockDev = this.blockDevs.get(obj);
                    this.blockDevs.remove(obj);
                    Log.debug((String)LOG_COMPONENT, (String)("removed block device " + blockDev.obj));
                    Hooks.chainOfResponsibility((HookContainer)this.luwrain, (String)"luwrain.linux.blockdev.removed", (Object[])new Object[]{new MapScriptObject(blockDev.createAttrMap())});
                    return;
                }
                return;
            }
            m = RE_PROP_CHANGED.matcher(line);
            if (m.find()) {
                String obj = m.group(1).trim();
                String iface = m.group(2).trim();
                if (obj.startsWith(OBJ_DRIVES) && this.disks.containsKey(obj)) {
                    this.activeDisk = this.disks.get(obj);
                    this.activeIface = iface;
                    Log.debug((String)LOG_COMPONENT, (String)("updating the disk " + obj + " with the interface " + this.activeIface));
                    return;
                }
                if (obj.startsWith(OBJ_BLOCK) && this.blockDevs.containsKey(obj)) {
                    this.activeBlockDev = this.blockDevs.get(obj);
                    this.activeIface = iface;
                    Log.debug((String)LOG_COMPONENT, (String)("updating the block device " + obj + " with the interface " + this.activeIface));
                    return;
                }
                Log.warning((String)LOG_COMPONENT, (String)("unrecognized updating line: " + line));
                return;
            }
            String l = line.trim();
            if (l.startsWith("org.freedesktop") && l.endsWith(":")) {
                this.activeIface = l.substring(0, l.length() - 1);
                return;
            }
            if (this.activeDisk != null) {
                this.activeDisk.onLine(this.activeIface, l);
            }
            if (this.activeBlockDev != null) {
                this.activeBlockDev.onLine(this.activeIface, l);
            }
        }
        catch (Throwable e) {
            Log.error((String)LOG_COMPONENT, (String)("unable to process a line of udisksctl output: " + e.getClass().getName() + ": " + e.getMessage()));
        }
    }

    private void init() throws IOException {
        String[] output;
        BashProcess p = new BashProcess("udisksctl dump", EnumSet.of(BashProcess.Flags.LOG_OUTPUT));
        p.run();
        int exitCode = p.waitFor();
        if (exitCode != 0) {
            throw new IOException("udisksctl dump returned exit code " + String.valueOf(exitCode));
        }
        for (String l : output = p.getOutput()) {
            String line = l.trim();
            if (line.startsWith("/org/freedesktop/") && line.endsWith(":")) {
                String obj = line.substring(0, line.length() - 1);
                if (obj.startsWith(OBJ_DRIVES)) {
                    this.activeDisk = new Disk(obj);
                    this.disks.put(obj, this.activeDisk);
                    this.activeBlockDev = null;
                    this.activeIface = null;
                    Log.debug((String)LOG_COMPONENT, (String)("adding disk " + obj));
                    continue;
                }
                if (obj.startsWith(OBJ_BLOCK)) {
                    this.activeBlockDev = new BlockDev(obj);
                    this.blockDevs.put(obj, this.activeBlockDev);
                    this.activeDisk = null;
                    this.activeIface = null;
                    Log.debug((String)LOG_COMPONENT, (String)("adding block device " + obj));
                    continue;
                }
                if (obj.startsWith(OBJ_MANAGER)) continue;
                Log.warning((String)LOG_COMPONENT, (String)("unrecognized initial object: " + line));
                continue;
            }
            if (line.startsWith("org.freedesktop.") && line.endsWith(":")) {
                this.activeIface = line.substring(0, line.length() - 1);
                continue;
            }
            if (this.activeIface == null) continue;
            if (this.activeDisk != null) {
                this.activeDisk.onLine(this.activeIface, line);
                continue;
            }
            if (this.activeBlockDev == null) continue;
            this.activeBlockDev.onLine(this.activeIface, line);
        }
        this.activeDisk = null;
        this.activeBlockDev = null;
        this.activeIface = null;
    }

    @Override
    public void onErrorLine(String line) {
        Log.error((String)LOG_COMPONENT, (String)("monitor error: " + line));
    }

    @Override
    public void onFinishing(int exitCode) {
        this.activeDisk = null;
        this.activeBlockDev = null;
        this.disks.clear();
        this.blockDevs.clear();
        if (!this.closed) {
            if (exitCode == 0) {
                Log.debug((String)LOG_COMPONENT, (String)"the monitor finished without errors");
            } else {
                Log.error((String)LOG_COMPONENT, (String)("the monitor finished with the exit code " + String.valueOf(exitCode)));
            }
        }
        this.closed = true;
    }

    private final class Disk {
        final String obj;
        String vendor = null;
        String model = null;
        String removable = null;
        String ejectable = null;

        Disk(String obj) {
            this.obj = obj;
        }

        void onLine(String iface, String line) {
            if (!iface.equals(UdisksCliMonitor.IFACE_DRIVE)) {
                return;
            }
            if (line.startsWith(UdisksCliMonitor.PREFIX_MODEL)) {
                this.model = line.substring(UdisksCliMonitor.PREFIX_MODEL.length()).trim();
            }
            if (line.startsWith(UdisksCliMonitor.PREFIX_VENDOR)) {
                this.vendor = line.substring(UdisksCliMonitor.PREFIX_VENDOR.length()).trim();
            }
            if (line.startsWith(UdisksCliMonitor.PREFIX_REMOVABLE)) {
                this.removable = line.substring(UdisksCliMonitor.PREFIX_REMOVABLE.length()).trim();
            }
            if (line.startsWith(UdisksCliMonitor.PREFIX_EJECTABLE)) {
                this.ejectable = line.substring(UdisksCliMonitor.PREFIX_EJECTABLE.length()).trim();
            }
        }

        boolean isReady() {
            return this.model != null && this.vendor != null && this.removable != null && this.ejectable != null;
        }

        Map<String, Object> createAttrMap() {
            HashMap<String, Object> d = new HashMap<String, Object>();
            d.put("obj", this.obj);
            d.put("model", this.model);
            d.put("vendor", this.vendor);
            return d;
        }
    }

    private final class BlockDev {
        final String obj;
        String device = null;
        String drive = null;
        String fsType = null;
        String mountPoints = "";

        BlockDev(String obj) {
            this.obj = obj;
        }

        void onLine(String iface, String line) {
            if (line.startsWith(UdisksCliMonitor.PREFIX_DEVICE)) {
                this.device = line.substring(UdisksCliMonitor.PREFIX_DEVICE.length()).trim();
            }
            if (line.startsWith(UdisksCliMonitor.PREFIX_DRIVE)) {
                this.drive = line.substring(UdisksCliMonitor.PREFIX_DRIVE.length()).trim();
            }
            if (line.startsWith(UdisksCliMonitor.PREFIX_FS_TYPE)) {
                this.fsType = line.substring(UdisksCliMonitor.PREFIX_FS_TYPE.length()).trim();
            }
            if (iface.equals(UdisksCliMonitor.IFACE_FILESYSTEM) && line.startsWith(UdisksCliMonitor.PREFIX_MOUNT_POINTS)) {
                this.mountPoints = line.substring(UdisksCliMonitor.PREFIX_MOUNT_POINTS.length()).trim();
            }
        }

        boolean isReady() {
            return this.device != null && this.drive != null && this.fsType != null;
        }

        Map<String, Object> createAttrMap() {
            HashMap<String, Object> d = new HashMap<String, Object>();
            d.put("obj", this.obj);
            d.put("device", this.device);
            d.put("drive", this.drive);
            d.put("fsType", this.fsType);
            d.put("mountPoints", this.mountPoints);
            return d;
        }
    }
}

